/*____________________________________________________________________________
	Copyright (C) 1997 Network Associates Inc. and affiliated companies.
	All rights reserved.

	$Id: pflPrefs.c,v 1.28 2001/03/28 00:24:57 dgal Exp $
____________________________________________________________________________*/

#include <stdio.h>
#include <string.h>

#include "pflPrefs.h"
#include "pgpPFLPriv.h"
#include "pgpMemoryMgrPriv.h"
#include "pgpMem.h"
#include "pgpDebug.h"
#include "pgpPFLErrors.h"
#include "pgpFileUtilities.h"
#include "pgpEndianConversion.h"
#include "pgpHex.h"
#include "pgpStrings.h"
#include "pgpMemoryIO.h"

typedef struct PGPPrefData	PGPPrefData;

typedef union PGPPrefDataUnion
{
	PGPBoolean		b;
	PGPUInt32		i;
	char *			str;
	PGPByte *		p;
	PGPPrefStruct *	strct;
	PGPPrefArray *	a;
	char *			raw;
} PGPPrefDataUnion;

struct PGPPrefData
{
	char *				prefComments;
	PGPPrefIndex		prefIndex;
	char *				prefName;
	PGPPrefType			prefType;
	PGPSize				dataLength;
	PGPPrefDataUnion	data;
	PGPUInt32			flags;

	PGPPrefData *	nextPrefData;
	PGPPrefData *	lastPrefData;
};

struct PGPPref 
{
	PGPMemoryMgrRef	memoryMgr;
	PFLFileSpecRef	prefFileSpec;
	PGPBoolean		dirty;

	PGPPrefData *	prefList;
	PGPPrefData *	prefListEnd;
} PGPPref;

#define IsLineEndChar(c)	((c == '\r') || (c == '\n'))
#define IsWhitespace(c)		((c == ' ') || (c == '\t'))

#define kPGPInvalidPrefIndex	-1

#ifndef kPGPLineEndString
#	if PGP_WIN32
#		define kPGPLineEndString	"\r\n"
#	elif PGP_MACINTOSH
#		define kPGPLineEndString	"\r"
#	elif PGP_UNIX
#		define kPGPLineEndString	"\n"
#	else
#		error Must set one of {PGP_WIN32, PGP_MACINTOSH, PGP_UNIX} = 1
#	endif
#endif


static PGPError sCreatePrefStruct(PGPMemoryMgrRef memoryMgr,
									PGPPrefStruct *templateStruct,
									PGPPrefStruct **newStruct);

static PGPError sCopyPrefFlags(PGPPrefRef prefRef,
					PGPPrefIndex prefIndex,
					PGPUInt32 flags);

static PGPError sFindPrefDataByIndex(PGPPrefRef prefRef,
					PGPPrefIndex prefIndex, 
					PGPPrefData **prefData);

static PGPError sFindPrefDataByName(PGPPrefRef prefRef,
					char *prefName, 
					PGPPrefData **prefData);

static PGPError sFreePref(PGPPrefRef prefRef);

static PGPError sAddPrefData(PGPPrefRef prefRef, 
					PGPPrefData *prefData);

static PGPError sRemovePrefData(PGPPrefRef prefRef,
					PGPPrefData *prefData);

static PGPError sGetNextPrefData(const PGPPrefData *currentPrefData,
					PGPPrefData **nextPrefData);

static PGPError sReadPrefData(PGPIORef prefIO, PGPPrefData *prefData,
					PGPUInt32 *arrayIndex);

static PGPError sWritePrefData(PGPIORef prefIO, PGPPrefData *prefData);

static PGPError sWritePrefLine(PGPIORef prefIO,
					PGPPrefData *prefData, PGPUInt32 index);

static PGPError sConvertRawData(PGPMemoryMgrRef memoryMgr, 
					PGPPrefData *oldPrefData, PGPPrefData *newPrefData,
					PGPUInt32 arrayIndex);

static PGPError sConvertRawDataToStruct(PGPMemoryMgrRef memoryMgr, 
					char *value, PGPUInt32 length, PGPPrefStruct *prefStruct);

static void sFreePrefData(PGPPrefData *prefData, PGPBoolean freeAll);

static PGPError sCopyStructDataToPref(PGPPrefRef prefRef, PGPBoolean readData,
					PGPPrefStruct *fromStruct, PGPPrefStruct *toStruct);

static void sStripQuotes(PGPUInt32 stringLen, char *string);

static PGPError sIORead(PGPIORef ioRef, PGPSize dataSize, 
					PGPBoolean crossPlatform, void *data);

static PGPError sIOWrite(PGPIORef ioRef, PGPSize dataSize, 
					PGPBoolean crossPlatform, const void *data);


PGPError PGPOpenPrefFile(PFLFileSpecRef prefFileSpec, 
							const PGPPrefDefinition *prefs,
							PGPUInt32 numPrefs,
							PGPPrefRef *prefRef)
{
	PGPUInt32				arrayIndex;
	PGPMemoryMgrRef			memoryMgr	= NULL;
	PGPIORef				prefIO;
	char *					newComments = NULL;
	char *					newName		= NULL;
	void *					newData		= NULL;
	PGPPrefData *			newPrefData = NULL;
	PGPPrefData *			oldPrefData = NULL;
	PGPError				err = kPGPError_NoErr;

	if (IsntNull(prefRef))
		*prefRef = NULL;

	PFLValidateFileSpec(prefFileSpec);

	memoryMgr = PFLGetFileSpecMemoryMgr( prefFileSpec );

	err = PGPNewMemoryPrefs(memoryMgr, prefs, numPrefs, prefRef);
	if (IsPGPError(err))
	{
		if (IsntNull(*prefRef))
		{
			sFreePref(*prefRef);
			*prefRef = NULL;
		}
		
		return err;
	}

	err = PFLCopyFileSpec(prefFileSpec, &((*prefRef)->prefFileSpec));
	if (IsPGPError(err))
	{
		sFreePref(*prefRef);
		*prefRef = NULL;
		return err;
	}

	err = PGPOpenFileSpec(	(*prefRef)->prefFileSpec, 
							kPFLFileOpenFlags_ReadOnly, 
							(PGPFileIORef *) &prefIO);
	
	if ((err != kPGPError_FileNotFound) && (err != kPGPError_FileOpFailed))
	{
		if (IsPGPError(err))
		{
			sFreePref(*prefRef);
			*prefRef = NULL;
			return err;
		}
		
		pgpAssert(PGPIOIsValid(prefIO));
		if (!PGPIOIsValid(prefIO))
		{
			sFreePref(*prefRef);
			*prefRef = NULL;
			return kPGPError_OutOfMemory;
		}
		
		while (!PGPIOIsAtEOF(prefIO) && IsntPGPError(err))
		{
			newPrefData = (PGPPrefData *) PGPNewData(memoryMgr,
											sizeof(PGPPrefData),
											kPGPMemoryMgrFlags_Clear);
			if (IsNull(newPrefData))
			{
				err = kPGPError_OutOfMemory;
				break;
			}

			err = sReadPrefData(prefIO, newPrefData, &arrayIndex);

			if (IsPGPError(err))
				break;
			
			if (IsntNull(newPrefData->prefName))
			{
				err = sFindPrefDataByName(*prefRef, newPrefData->prefName,
						&oldPrefData);

				if (IsntPGPError(err))
				{
					if (IsntNull(oldPrefData))
					{
						sConvertRawData(memoryMgr, oldPrefData, newPrefData,
							arrayIndex);

						PGPFreeData(newPrefData->prefName);
						PGPFreeData(newPrefData);
						newPrefData = NULL;
					}
					else
					{
						if (arrayIndex > 0)
						{
							char number[20];

							sprintf(number, "%d", arrayIndex);
							strcat(newPrefData->prefName, number);
						}

						sAddPrefData(*prefRef, newPrefData);
					}
				}
				else
				{
					sFreePrefData(newPrefData, TRUE);
					PGPFreeData(newPrefData);
					newPrefData = NULL;
				}
			}
			else
			{
				sFreePrefData(newPrefData, TRUE);
				PGPFreeData(newPrefData);
				newPrefData = NULL;
			}
		}
		
		if (err == kPGPError_EOF)
		{
			if (IsntNull(newPrefData))
			{
				sFreePrefData(newPrefData, TRUE);
				PGPFreeData(newPrefData);
			}

			err = kPGPError_NoErr;
		}
		else if (IsPGPError(err))
		{
			pgpAssertNoErr(err);
			sFreePref(*prefRef);
			*prefRef = NULL;
		}
		
		PGPFreeIO(prefIO);
	}
	else
		err = kPGPError_NoErr;

	if (IsntNull(*prefRef))
		(*prefRef)->dirty = FALSE;

	return err;
}


PGPError PGPSavePrefFile(PGPPrefRef prefRef)
{
	PGPPrefData *	currentPrefData;
	PGPIORef		prefIO;
	PGPError		err;
	PGPBoolean		fileExists;
	
	PGPValidatePref(prefRef);

	/* If this is a memory pref, do nothing */
	if (IsNull(prefRef->prefFileSpec))
		return kPGPError_NoErr;

	/* Create the file if it does not exist */
	err = PFLFileSpecExists( prefRef->prefFileSpec, &fileExists );
	if( IsntPGPError( err ) )
	{
		if( fileExists )
		{
			/* If no changes have been been, don't save the file */

			if (!prefRef->dirty)
				return kPGPError_NoErr;

			(void) PFLFileSpecDelete( prefRef->prefFileSpec );
		}
		err = PFLFileSpecCreate(prefRef->prefFileSpec);
	}
	
	pgpAssertNoErr(err);
	if (IsPGPError(err))
		return err;

	err = PGPOpenFileSpec(	prefRef->prefFileSpec, 
							kPFLFileOpenFlags_ReadWrite, 
							(PGPFileIORef *) &prefIO);
	pgpAssertNoErr(err);
	if (IsPGPError(err))
		return err;

	currentPrefData = prefRef->prefList;

	while (IsntNull(currentPrefData))
	{
		err = sWritePrefData(prefIO, currentPrefData);

		pgpAssertNoErr(err);
		if (IsPGPError(err))
			break;

		sGetNextPrefData(currentPrefData, &currentPrefData);
	}
	
	PGPFreeIO(prefIO);
	prefRef->dirty = FALSE;
	return err;
}


PGPError PGPGetPrefFileSpec(PGPPrefRef prefRef,
								PFLFileSpecRef *prefFileSpec)
{
	PGPError err = kPGPError_NoErr;

	if (IsntNull(prefFileSpec))
		*prefFileSpec = NULL;

	PGPValidatePref(prefRef);
	PGPValidatePtr(prefFileSpec);

	/* If this is a memory pref, the file spec is NULL */
	if (IsNull(prefRef->prefFileSpec))
		return kPGPError_NoErr;

	err = PFLCopyFileSpec(prefRef->prefFileSpec, prefFileSpec);
	if (IsPGPError(err))
		*prefFileSpec = NULL;

	return err;
}


PGPError PGPNewMemoryPrefs(PGPMemoryMgrRef memoryMgr, 
							const PGPPrefDefinition *prefs,
							PGPUInt32 numPrefs,
							PGPPrefRef *prefRef)
{
	PGPUInt32				index;
	PGPUInt32				arrayIndex;
	PGPPrefArray *			prefArray = NULL;
	PGPPrefArrayElement *	copyElement = NULL;
	PGPPrefArrayElement *	newElement = NULL;
	PGPPrefData *			newPrefData = NULL;
	PGPError				err = kPGPError_NoErr;

	if (IsntNull(prefRef))
		*prefRef = NULL;

	PGPValidatePtr(prefRef);
	PGPValidatePtr(prefs);
	PGPValidateParam(numPrefs > 0);

	pgpAssert(PGPMemoryMgrIsValid(memoryMgr));
	if (!PGPMemoryMgrIsValid(memoryMgr))
		return kPGPError_BadParams;

	*prefRef = (PGPPrefRef) PGPNewData(memoryMgr, 
								sizeof(PGPPref), 
								kPGPMemoryMgrFlags_Clear);

	pgpAssert(IsntNull(*prefRef));
	if (IsNull(*prefRef))
		return kPGPError_OutOfMemory;

	(*prefRef)->memoryMgr		= memoryMgr;
	(*prefRef)->prefFileSpec	= NULL;
	(*prefRef)->prefList		= NULL;
	(*prefRef)->prefListEnd		= NULL;

	for (index=0; index<numPrefs; index++)
	{
		newPrefData = (PGPPrefData *) PGPNewData(memoryMgr,
										sizeof(PGPPrefData),
										kPGPMemoryMgrFlags_Clear);
		
		if (IsNull(newPrefData))
		{
			err = kPGPError_OutOfMemory;
			break;
		}
		
		newPrefData->prefComments = NULL;
		newPrefData->prefIndex = prefs[index].index;
		newPrefData->prefType = prefs[index].type;
		
		newPrefData->prefName = (char *) PGPNewData(memoryMgr,
											strlen(prefs[index].name)+1,
											kPGPMemoryMgrFlags_Clear);
		
		if (IsNull(newPrefData->prefName))
		{
			err = kPGPError_OutOfMemory;
			break;
		}
		
		pgpCopyMemory(prefs[index].name, newPrefData->prefName,
			strlen(prefs[index].name) + 1);
		
		switch (prefs[index].type)
		{
		case kPGPPrefType_Boolean:
			{
				newPrefData->dataLength = sizeof(PGPBoolean);
				newPrefData->data.b = (PGPBoolean) (size_t) 
										prefs[index].data;
				break;
			}
		case kPGPPrefType_Number:
			{
				newPrefData->dataLength = sizeof(PGPUInt32);
				newPrefData->data.i = (PGPUInt32) (size_t)
										prefs[index].data; 
				break;
			}
			
		case kPGPPrefType_String:
			if (IsntNull(prefs[index].data))
			{
				newPrefData->dataLength = strlen((char *) prefs[index].data) 
											+ 1;
				
				newPrefData->data.str = PGPNewData(memoryMgr,
											newPrefData->dataLength,
											kPGPMemoryMgrFlags_Clear);
				
				if (IsNull(newPrefData->data.str))
				{
					err = kPGPError_OutOfMemory;
					break;
				}
				
				pgpCopyMemory(prefs[index].data, newPrefData->data.str, 
					newPrefData->dataLength);
			}
			else
			{
				newPrefData->dataLength = 0;
				newPrefData->data.str = NULL;
			}
			break;
			
		case kPGPPrefType_Byte:
			if (IsntNull(prefs[index].data))
			{
				newPrefData->dataLength = prefs[index].size;
				newPrefData->data.p = PGPNewData(memoryMgr,
										newPrefData->dataLength,
										kPGPMemoryMgrFlags_Clear);
				
				if (IsNull(newPrefData->data.p))
				{
					err = kPGPError_OutOfMemory;
					break;
				}
				
				pgpCopyMemory(prefs[index].data, newPrefData->data.p,
					newPrefData->dataLength);
			}
			else
			{
				newPrefData->dataLength = 0;
				newPrefData->data.p = NULL;
			}
			break;

		case kPGPPrefType_Struct:
			newPrefData->dataLength = sizeof(PGPPrefStruct);
			newPrefData->data.strct = (PGPPrefStruct *) 
										PGPNewData(memoryMgr,
											sizeof(PGPPrefStruct),
											kPGPMemoryMgrFlags_Clear);

			if (IsNull(newPrefData->data.strct))
			{
				err = kPGPError_OutOfMemory;
				break;
			}

			err = sCopyStructDataToPref(*prefRef, TRUE,
					(PGPPrefStruct *) prefs[index].data,
					newPrefData->data.strct);
			break;

		case kPGPPrefType_Array:
			newPrefData->dataLength = sizeof(PGPPrefArray);
			prefArray = (PGPPrefArray *) prefs[index].data;

			newPrefData->data.a = (PGPPrefArray *) PGPNewData(memoryMgr,
													sizeof(PGPPrefArray),
													kPGPMemoryMgrFlags_Clear);

			if (IsNull(newPrefData->data.a))
			{
				err = kPGPError_OutOfMemory;
				break;
			}

			pgpCopyMemory(prefArray, newPrefData->data.a, 
				sizeof(PGPPrefArray));

			if (prefArray->numElements > 0)
			{
				newPrefData->data.a->elements = PGPNewData(memoryMgr,
												sizeof(PGPPrefArrayElement) *
												prefArray->numElements,
												kPGPMemoryMgrFlags_Clear);

				if (IsNull(newPrefData->data.a->elements))
				{
					PGPFreeData(newPrefData->data.a);
					err = kPGPError_OutOfMemory;
					break;
				}
			}
			
			if (prefArray->type == kPGPPrefType_Struct)
			{
				PGPValidatePtr(prefArray->templateStruct);

				newPrefData->data.a->templateStruct = (PGPPrefStruct *)
											PGPNewData(memoryMgr,
												sizeof(PGPPrefStruct),
												kPGPMemoryMgrFlags_Clear);
			
				if (IsNull(newPrefData->data.a->templateStruct))
				{
					err = kPGPError_OutOfMemory;
					break;
				}
			
				err = sCopyStructDataToPref(*prefRef, TRUE,
						prefArray->templateStruct,
						newPrefData->data.a->templateStruct);
			
				if (IsPGPError(err))
					break;
			}
			
			for (arrayIndex=0; arrayIndex<prefArray->numElements; arrayIndex++)
			{
				copyElement = &(prefArray->elements[arrayIndex]);
				newElement = &(newPrefData->data.a->elements[arrayIndex]);

				newElement->size = copyElement->size;

				switch (prefArray->type)
				{
				case kPGPPrefType_Boolean:
				case kPGPPrefType_Number:
					newElement->data = copyElement->data; 
					break;

				case kPGPPrefType_String:
					if (IsntNull(copyElement->data))
						newElement->size = strlen(copyElement->data) + 1;
					else
						newElement->size = 0;

				case kPGPPrefType_Byte:
					if (IsntNull(copyElement->data))
					{
						newElement->data = PGPNewData(memoryMgr,
												newElement->size,
												kPGPMemoryMgrFlags_Clear);
					
						if (IsNull(newElement->data))
						{
							err = kPGPError_OutOfMemory;
							break;
						}

						pgpCopyMemory(copyElement->data, newElement->data,
							newElement->size);
					}
					else
						newElement->data = NULL; 
					
					break;

				case kPGPPrefType_Struct:
					newElement->size = sizeof(PGPPrefStruct);
					newElement->data = PGPNewData(memoryMgr,
											sizeof(PGPPrefStruct),
											kPGPMemoryMgrFlags_Clear);
					
					if (IsNull(newElement->data))
					{
						err = kPGPError_OutOfMemory;
						break;
					}

					err = sCopyStructDataToPref(*prefRef, TRUE,
							(PGPPrefStruct *) copyElement->data,
							(PGPPrefStruct *) newElement->data);
					break;
				}
			}
			break;
		}
	
		if (IsntPGPError(err))
			sAddPrefData(*prefRef, newPrefData);
		else
		{
			sFreePrefData(newPrefData, TRUE);
			PGPFreeData(newPrefData);
			break;
		}
	}

	if (IsntNull(*prefRef))
		(*prefRef)->dirty = FALSE;

	return err;
}


PGPError PGPAddPrefs(PGPPrefRef prefsToAdd, PGPPrefRef prefRef)
{
	PGPPrefData *addData = NULL;
	PGPError err = kPGPError_NoErr;

	PGPValidatePref(prefsToAdd);
	PGPValidatePref(prefRef);

	addData = prefsToAdd->prefList;
	while (IsntNull(addData) && IsntPGPError(err))
	{
		switch (addData->prefType)
		{
		case kPGPPrefType_Boolean:
			err = PGPSetPrefData(prefRef, addData->prefIndex, 
					addData->dataLength, &(addData->data.b));
			break;

		case kPGPPrefType_Number:
			err = PGPSetPrefData(prefRef, addData->prefIndex, 
					addData->dataLength, &(addData->data.i));
			break;

		case kPGPPrefType_String:
			err = PGPSetPrefData(prefRef, addData->prefIndex, 
					addData->dataLength, addData->data.str);
			break;

		case kPGPPrefType_Byte:
			err = PGPSetPrefData(prefRef, addData->prefIndex, 
					addData->dataLength, addData->data.p);
			break;

		case kPGPPrefType_Struct:
			err = PGPSetPrefData(prefRef, addData->prefIndex, 
					addData->dataLength, addData->data.strct);
			break;

		case kPGPPrefType_Array:
			err = PGPSetPrefData(prefRef, addData->prefIndex, 
					addData->dataLength, addData->data.a);
			break;
		}

		if (IsntPGPError(err))
			err = sCopyPrefFlags(prefRef, addData->prefIndex, 
					addData->flags);

		if (err == kPGPError_PrefNotFound)
			err = kPGPError_NoErr;

		if (IsntPGPError(err))
			err = sGetNextPrefData(addData, &addData);
	}

	return err;
}


PGPError PGPFreePrefs(PGPPrefRef prefRef)
{
	PGPValidatePref(prefRef);

	sFreePref(prefRef);
	return kPGPError_NoErr;
}


PGPError PGPPeekPrefMemoryMgr(PGPPrefRef prefRef,
								PGPMemoryMgrRef *memoryMgr)
{
	if (IsntNull(memoryMgr))
		*memoryMgr = NULL;

	PGPValidatePref(prefRef);
	PGPValidatePtr(memoryMgr);

	*memoryMgr = prefRef->memoryMgr;
	return kPGPError_NoErr;
}

 
PGPError PGPGetPrefData(PGPPrefRef prefRef, 
							PGPPrefIndex prefIndex, 
							PGPSize *dataLength, 
							void **inBuffer)
{
	PGPPrefData *			currentPrefData;
	PGPPrefArray *			prefArray;
	PGPPrefArrayElement *	copyElement;
	PGPPrefArrayElement *	newElement;
	PGPUInt32				index;
	PGPError				err;

	if (IsntNull(dataLength))
		*dataLength = 0;
	if (IsntNull(inBuffer))
		*inBuffer = NULL;

	PGPValidatePtr(inBuffer);
	PGPValidatePtr(dataLength);
	PGPValidatePref(prefRef);

	err = sFindPrefDataByIndex(prefRef, prefIndex, &currentPrefData);
	pgpAssertNoErr(err);
	if (IsPGPError(err))
		return err;

	if (IsNull(currentPrefData))
		return kPGPError_PrefNotFound;

	if (currentPrefData->dataLength > 0)
	{
		*inBuffer	= PGPNewData(prefRef->memoryMgr,
						currentPrefData->dataLength,
						kPGPMemoryMgrFlags_Clear);

		if (IsNull(*inBuffer))
			return kPGPError_OutOfMemory;
		else
			*dataLength	= currentPrefData->dataLength;
	}
	else
		return kPGPError_PrefNotFound;

	switch (currentPrefData->prefType)
	{
	case kPGPPrefType_Boolean:
		pgpCopyMemory(&(currentPrefData->data.b), *inBuffer, 
			sizeof(PGPBoolean));
		break;

	case kPGPPrefType_Number:
		pgpCopyMemory(&(currentPrefData->data.i), *inBuffer, 
			sizeof(PGPUInt32));
		break;

	case kPGPPrefType_String:
		if (IsntNull(currentPrefData->data.str))
			pgpCopyMemory(currentPrefData->data.str, *inBuffer, 
				currentPrefData->dataLength);
		break;

	case kPGPPrefType_Byte:
		if (IsntNull(currentPrefData->data.p))
			pgpCopyMemory(currentPrefData->data.p, *inBuffer, 
				currentPrefData->dataLength);
		break;

	case kPGPPrefType_Struct:
		if (IsntNull(currentPrefData->data.strct))
			err = sCopyStructDataToPref(prefRef, TRUE, 
					currentPrefData->data.strct, (PGPPrefStruct *) *inBuffer);
		break;

	case kPGPPrefType_Array:
		if (IsNull(currentPrefData->data.a))
			break;

		prefArray = (PGPPrefArray *) *inBuffer;
		prefArray->numElements = currentPrefData->data.a->numElements;
		prefArray->type = currentPrefData->data.a->type;

		if (prefArray->numElements > 0)
		{
			prefArray->elements = (PGPPrefArrayElement *) 
										PGPNewData(prefRef->memoryMgr,
											sizeof(PGPPrefArrayElement) *
											prefArray->numElements,
											kPGPMemoryMgrFlags_Clear);

			if (IsNull(prefArray->elements))
			{
				err = kPGPError_OutOfMemory;
				PGPFreeData(*inBuffer);
				*inBuffer = NULL;
				break;
			}
		}

		if (prefArray->type == kPGPPrefType_Struct)
		{
			prefArray->templateStruct = (PGPPrefStruct *) 
										PGPNewData(prefRef->memoryMgr,
											sizeof(PGPPrefStruct),
											kPGPMemoryMgrFlags_Clear);

			if (IsNull(prefArray->templateStruct))
			{
				err = kPGPError_OutOfMemory;

				if (IsntNull(prefArray->elements))
					PGPFreeData(prefArray->elements);

				PGPFreeData(*inBuffer);
				*inBuffer = NULL;
				break;
			}

			err = sCopyStructDataToPref(prefRef, TRUE,
					currentPrefData->data.a->templateStruct,
					prefArray->templateStruct);

			if (IsPGPError(err))
			{
				PGPFreeData(prefArray->templateStruct);

				if (IsntNull(prefArray->elements))
					PGPFreeData(prefArray->elements);

				PGPFreeData(*inBuffer);
				*inBuffer = NULL;
				break;
			}
		}

		for (index=0; index<currentPrefData->data.a->numElements; index++)
		{
			copyElement = &(currentPrefData->data.a->elements[index]);
			newElement = &(prefArray->elements[index]);

			newElement->size = copyElement->size;

			switch (prefArray->type)
			{
			case kPGPPrefType_Boolean:
			case kPGPPrefType_Number:
				newElement->data = copyElement->data; 
				break;
				
			case kPGPPrefType_String:
				if (IsntNull(copyElement->data))
					newElement->size = strlen(copyElement->data) + 1;
				else
					newElement->size = 0;
				
			case kPGPPrefType_Byte:
				if (IsntNull(copyElement->data))
				{
					newElement->data = PGPNewData(prefRef->memoryMgr,
											newElement->size,
											kPGPMemoryMgrFlags_Clear);
					
					if (IsNull(newElement->data))
					{
						err = kPGPError_OutOfMemory;
						break;
					}
					
					pgpCopyMemory(copyElement->data, newElement->data,
						newElement->size);
				}
				else
					newElement->data = NULL; 
				
				break;
				
			case kPGPPrefType_Struct:
				newElement->size = sizeof(PGPPrefStruct);
				newElement->data = PGPNewData(prefRef->memoryMgr,
									sizeof(PGPPrefStruct),
									kPGPMemoryMgrFlags_Clear);
				
				if (IsNull(newElement->data))
				{
					err = kPGPError_OutOfMemory;
					break;
				}
				
				err = sCopyStructDataToPref(prefRef, TRUE,
						(PGPPrefStruct *) copyElement->data,
						(PGPPrefStruct *) newElement->data);
				break;
			}
		}
		break;
	}

	return err;
}


PGPError PGPSetPrefData(PGPPrefRef prefRef, 
							PGPPrefIndex prefIndex, 
							PGPSize dataLength, 
							const void *outBuffer)
{
	PGPPrefData *			currentPrefData;
	PGPPrefArray *			prefArray;
	PGPPrefArrayElement *	copyElement;
	PGPPrefArrayElement *	newElement;
	PGPUInt32				index;
	PGPError				err;
	
	PGPValidatePref(prefRef);

	err = sFindPrefDataByIndex(prefRef, prefIndex, &currentPrefData);
	pgpAssertNoErr(err);
	if (IsPGPError(err))
		return err;

	if (IsNull(currentPrefData))
		return kPGPError_PrefNotFound;

	if (currentPrefData->prefType != kPGPPrefType_Byte)
	{
		PGPValidatePtr(outBuffer);
		PGPValidateParam(dataLength > 0);
	}

	switch (currentPrefData->prefType)
	{
	case kPGPPrefType_Boolean:
		if (!prefRef->dirty)
		{
			if (!pgpMemoryEqual(outBuffer, &(currentPrefData->data.b), 
				dataLength))
				prefRef->dirty = TRUE;
		}

		if (prefRef->dirty)
			pgpCopyMemory(outBuffer, &(currentPrefData->data.b), dataLength);
		break;

	case kPGPPrefType_Number:
		if (!prefRef->dirty)
		{
			if (!pgpMemoryEqual(outBuffer, &(currentPrefData->data.i), 
				dataLength))
				prefRef->dirty = TRUE;
		}

		if (prefRef->dirty)
			pgpCopyMemory(outBuffer, &(currentPrefData->data.i), dataLength);
		break;

	case kPGPPrefType_String:
		if (currentPrefData->dataLength != dataLength)
			prefRef->dirty = TRUE;

		if (IsntNull(currentPrefData->data.str))
		{
			if (!prefRef->dirty)
			{
				if (!pgpMemoryEqual(outBuffer, currentPrefData->data.str,
					dataLength))
					prefRef->dirty = TRUE;
			}
		}

		if (prefRef->dirty)
		{
			if (IsntNull(currentPrefData->data.str))
				PGPFreeData(currentPrefData->data.str);
		
			currentPrefData->data.str = PGPNewData(prefRef->memoryMgr,
											dataLength,
											kPGPMemoryMgrFlags_Clear);
			
			if (IsNull(currentPrefData->data.str))
			{
				sRemovePrefData(prefRef, currentPrefData);
				return kPGPError_OutOfMemory;
			}
			
			pgpCopyMemory(outBuffer, currentPrefData->data.str, dataLength);
			sStripQuotes(dataLength, currentPrefData->data.str);
			dataLength = strlen(currentPrefData->data.str) + 1;
		}
		break;

	case kPGPPrefType_Byte:
		if (currentPrefData->dataLength != dataLength)
			prefRef->dirty = TRUE;

		if (IsntNull(currentPrefData->data.p) && IsntNull(outBuffer) &&
			!prefRef->dirty)
		{
			if (!pgpMemoryEqual(outBuffer, currentPrefData->data.p, dataLength))
				prefRef->dirty = TRUE;
		}

		if (prefRef->dirty)
		{
			if (IsntNull(currentPrefData->data.p))
			{
				PGPFreeData(currentPrefData->data.p);
				currentPrefData->data.p = NULL;
			}
			
			if (IsntNull(outBuffer))
			{
				currentPrefData->data.p = PGPNewData(prefRef->memoryMgr,
											dataLength,
											kPGPMemoryMgrFlags_Clear);
				
				if (IsNull(currentPrefData->data.p))
				{
					sRemovePrefData(prefRef, currentPrefData);
					return kPGPError_OutOfMemory;
				}
				
				pgpCopyMemory(outBuffer, currentPrefData->data.p, dataLength);
			}
		}
		break;

	case kPGPPrefType_Struct:
		err = sCopyStructDataToPref(prefRef, FALSE,
				(PGPPrefStruct *) outBuffer, currentPrefData->data.strct);
		break;

	case kPGPPrefType_Array:
		prefArray = (PGPPrefArray *) outBuffer;
		if (prefArray->type != currentPrefData->data.a->type)
		{
			err = kPGPError_BadParams;
			break;
		}

		if (prefArray->numElements != currentPrefData->data.a->numElements)
		{
			prefRef->dirty = TRUE;
			PGPFreePrefArray(currentPrefData->data.a);

			err = PGPCreatePrefArray(prefRef, prefArray->type, 
					prefArray->numElements, prefArray->templateStruct,
					&(currentPrefData->data.a));

			if (IsPGPError(err))
				break;
		}

		for (index=0; index<prefArray->numElements; index++)
		{
			copyElement = &(prefArray->elements[index]);
			newElement = &(currentPrefData->data.a->elements[index]);
			
			if ((newElement->size != copyElement->size) && 
				((prefArray->type == kPGPPrefType_String) || 
				(prefArray->type == kPGPPrefType_Byte)))
			{
				prefRef->dirty = TRUE;
			}
			
			newElement->size = copyElement->size;

			switch (prefArray->type)
			{
			case kPGPPrefType_Boolean:
			case kPGPPrefType_Number:
				if (newElement->data != copyElement->data)
				{
					prefRef->dirty = TRUE;
					newElement->data = copyElement->data;
				}
				break;
				
			case kPGPPrefType_String:
				if (IsntNull(copyElement->data))
					newElement->size = strlen(copyElement->data) + 1;
				else
					newElement->size = 0;
				
			case kPGPPrefType_Byte:
				if (IsntNull(newElement->data))
				{
					if (IsntNull(copyElement->data) && !prefRef->dirty)
					{
						if (!pgpMemoryEqual(newElement->data, 
							copyElement->data, copyElement->size))
							prefRef->dirty = TRUE;
					}
					else
						prefRef->dirty = TRUE;
				}
				else if (IsntNull(copyElement->data))
					prefRef->dirty = TRUE;

				if (prefRef->dirty)
				{
					if (IsntNull(newElement->data))
						PGPFreeData(newElement->data);
					
					if (IsntNull(copyElement->data))
					{
						newElement->data = PGPNewData(prefRef->memoryMgr,
												newElement->size,
												kPGPMemoryMgrFlags_Clear);
						
						if (IsNull(newElement->data))
						{
							err = kPGPError_OutOfMemory;
							break;
						}
						
						pgpCopyMemory(copyElement->data, newElement->data,
							newElement->size);
					}
					else
						newElement->data = NULL; 
				}
				
				break;
				
			case kPGPPrefType_Struct:
				if (IsNull(newElement->data))
				{
					prefRef->dirty = TRUE;
					newElement->size = sizeof(PGPPrefStruct);
					newElement->data = PGPNewData(prefRef->memoryMgr,
											sizeof(PGPPrefStruct),
											kPGPMemoryMgrFlags_Clear);
				
					if (IsNull(newElement->data))
					{
						err = kPGPError_OutOfMemory;
						break;
					}
				}
				
				err = sCopyStructDataToPref(prefRef, FALSE,
						(PGPPrefStruct *) copyElement->data,
						(PGPPrefStruct *) newElement->data);
				break;
			}
		}
		break;
	}

	currentPrefData->dataLength = dataLength;
	return err;
}


PGPError PGPClearPrefData(PGPPrefRef prefRef, 
							PGPPrefIndex prefIndex)
{
	PGPPrefData *	currentPrefData;
	PGPPrefStruct *	tempStruct;
	PGPPrefArray *	tempArray;
	PGPError		err;
	
	PGPValidatePref(prefRef);

	err = sFindPrefDataByIndex(prefRef, prefIndex, &currentPrefData);
	pgpAssertNoErr(err);
	if (IsPGPError(err))
		return err;

	if (IsNull(currentPrefData))
		return kPGPError_PrefNotFound;

	prefRef->dirty = TRUE;

	switch (currentPrefData->prefType)
	{
	case kPGPPrefType_Boolean:
		currentPrefData->data.b = FALSE;
		break;

	case kPGPPrefType_Number:
		currentPrefData->data.i = 0;
		break;

	case kPGPPrefType_String:
	case kPGPPrefType_Byte:
		sFreePrefData(currentPrefData, FALSE);
		break;

	case kPGPPrefType_Struct:
		tempStruct = currentPrefData->data.strct;
		PGPCreatePrefStruct(prefRef, tempStruct, 
			&(currentPrefData->data.strct));
		PGPFreePrefStruct(tempStruct);
		break;

	case kPGPPrefType_Array:
		tempArray = currentPrefData->data.a;
		PGPCreatePrefArray(prefRef, tempArray->type, 0, 
			tempArray->templateStruct, &(currentPrefData->data.a));
		PGPFreePrefArray(tempArray);
		break;
	}

	return kPGPError_NoErr;
}


PGPError PGPRemovePref(PGPPrefRef prefRef, 
						   PGPPrefIndex prefIndex)
{
	PGPPrefData *	currentPrefData;
	PGPError		err;
	
	PGPValidatePref(prefRef);

	err = sFindPrefDataByIndex(prefRef, prefIndex, &currentPrefData);
	pgpAssertNoErr(err);
	if (IsPGPError(err))
		return err;

	if (IsNull(currentPrefData))
		return kPGPError_PrefNotFound;

	sRemovePrefData(prefRef, currentPrefData);
	prefRef->dirty = TRUE;
	return kPGPError_NoErr;
}


PGPError PGPCreatePrefStruct(PGPPrefRef prefRef,
								PGPPrefStruct *templateStruct,
								PGPPrefStruct **newStruct)
{
	PGPValidatePref(prefRef);

	return sCreatePrefStruct(prefRef->memoryMgr, templateStruct, newStruct);
}

static PGPError sCreatePrefStruct(PGPMemoryMgrRef memoryMgr,
									PGPPrefStruct *templateStruct,
									PGPPrefStruct **newStruct)
{
	PGPUInt32 index;

	if (IsntNull(newStruct))
		*newStruct = NULL;

	PGPValidatePtr(templateStruct);
	PGPValidatePtr(newStruct);
	PGPValidateParam(templateStruct->numMembers > 0);

	*newStruct = (PGPPrefStruct *) PGPNewData(memoryMgr,
										sizeof(PGPPrefStruct),
										kPGPMemoryMgrFlags_Clear);

	if (IsNull(*newStruct))
		return kPGPError_OutOfMemory;

	(*newStruct)->numMembers = templateStruct->numMembers;

	(*newStruct)->members = (PGPPrefStructMember *) 
								PGPNewData(memoryMgr,
									sizeof(PGPPrefStructMember) *
									templateStruct->numMembers,
									kPGPMemoryMgrFlags_Clear);

	if (IsNull((*newStruct)->members))
	{
		PGPFreeData(*newStruct);
		*newStruct = NULL;
		return kPGPError_OutOfMemory;
	}

	for (index=0; index<templateStruct->numMembers; index++)
	{
		(*newStruct)->members[index].type = 
			templateStruct->members[index].type;

		(*newStruct)->members[index].data = NULL;
		(*newStruct)->members[index].size = 0;
	}

	return kPGPError_NoErr;
}


PGPError PGPFreePrefStruct(PGPPrefStruct *prefStruct)
{
	PGPUInt32 index;

	PGPValidatePtr(prefStruct);
	
	pgpAssertMsg(!(prefStruct->dirty), "Pref struct data was"
		" changed, but changes were not saved with PGPSetPrefStruct");

	for (index=0; index<prefStruct->numMembers; index++)
	{
		switch (prefStruct->members[index].type)
		{
		case kPGPPrefType_String:
		case kPGPPrefType_Byte:
			if (IsntNull(prefStruct->members[index].data))
				PGPFreeData(prefStruct->members[index].data);
			break;
		}
	}
	
	PGPFreeData(prefStruct->members);
	PGPFreeData(prefStruct);
	
	return kPGPError_NoErr;
}


PGPError PGPCreatePrefArray(PGPPrefRef prefRef,
								PGPPrefType arrayType,
								PGPUInt32 arraySize,
								PGPPrefStruct *templateStruct,
								PGPPrefArray **prefArray)
{
	PGPError err = kPGPError_NoErr;

	if (IsntNull(prefArray))
		*prefArray = NULL;

	PGPValidatePref(prefRef);
	PGPValidatePtr(prefArray);

	*prefArray = (PGPPrefArray *) PGPNewData(prefRef->memoryMgr,
										sizeof(PGPPrefArray),
										kPGPMemoryMgrFlags_Clear);

	if (IsNull(*prefArray))
		return kPGPError_OutOfMemory;

	(*prefArray)->numElements = arraySize;
	(*prefArray)->type = arrayType;
	(*prefArray)->dirty = FALSE;

	if (arraySize > 0)
	{
		(*prefArray)->elements = (PGPPrefArrayElement *) 
									PGPNewData(prefRef->memoryMgr,
										sizeof(PGPPrefArrayElement) *
										arraySize,
										kPGPMemoryMgrFlags_Clear);

		if (IsNull((*prefArray)->elements))
		{
			PGPFreeData(*prefArray);
			*prefArray = NULL;
			return kPGPError_OutOfMemory;
		}
	}

	if (arrayType == kPGPPrefType_Struct)
	{
		PGPValidatePtr(templateStruct);
		PGPValidateParam(templateStruct->numMembers > 0);

		(*prefArray)->templateStruct = (PGPPrefStruct *) 
										PGPNewData(prefRef->memoryMgr,
											sizeof(PGPPrefStruct),
											kPGPMemoryMgrFlags_Clear);

		if (IsNull((*prefArray)->templateStruct))
		{
			if (IsntNull((*prefArray)->elements))
				PGPFreeData((*prefArray)->elements);

			PGPFreeData(*prefArray);
			*prefArray = NULL;
			return kPGPError_OutOfMemory;
		}

		err = sCopyStructDataToPref(prefRef, TRUE, templateStruct,
				(*prefArray)->templateStruct);

		if (IsPGPError(err))
		{
			if (IsntNull((*prefArray)->elements))
				PGPFreeData((*prefArray)->elements);

			PGPFreeData((*prefArray)->templateStruct);
			PGPFreeData(*prefArray);
			*prefArray = NULL;
			return err;
		}
	}

	return kPGPError_NoErr;
}


PGPError PGPFreePrefArray(PGPPrefArray *prefArray)
{
	PGPUInt32 index;
	PGPPrefStruct *prefStruct;

	PGPValidatePtr(prefArray);
	
	pgpAssertMsg(!(prefArray->dirty), "Pref array data or flags were"
		" changed, but changes were not saved with PGPSetPrefArray");

	switch (prefArray->type)
	{
	case kPGPPrefType_String:
	case kPGPPrefType_Byte:
		for (index=0; index<prefArray->numElements; index++)
		{
			if (IsntNull(prefArray->elements[index].data))
				PGPFreeData(prefArray->elements[index].data);
		}
		break;
		
	case kPGPPrefType_Struct:
		PGPFreePrefStruct(prefArray->templateStruct);

		for (index=0; index<prefArray->numElements; index++)
		{
			prefStruct = (PGPPrefStruct *) prefArray->elements[index].data;
			PGPFreePrefStruct(prefStruct);
		}
		break;
	}
	
	if (IsntNull(prefArray->elements))
		PGPFreeData(prefArray->elements);

	PGPFreeData(prefArray);

	return kPGPError_NoErr;
}


static PGPError sExportPrefData(PGPIORef memIO, PGPPrefType type,
									PGPSize dataLength, PGPPrefDataUnion data)
{
	PGPSize bytes;
	PGPUInt32 index;
	PGPPrefDataUnion strctData;
	PGPError err = kPGPError_NoErr;

	switch (type)
	{
	case kPGPPrefType_Boolean:
		err = sIOWrite(memIO, sizeof(PGPBoolean), FALSE, &(data.b));
		break;
		
	case kPGPPrefType_Number:
		err = sIOWrite(memIO, sizeof(PGPUInt32), TRUE, &(data.i));
		break;
		
	case kPGPPrefType_String:
		if (IsntNull(data.str))
		{
			bytes = strlen(data.str) + 1;
			
			err = sIOWrite(memIO, sizeof(PGPSize), TRUE, &bytes);
			
			if (IsntPGPError(err))
				err = sIOWrite(memIO, bytes, FALSE, data.str);
		}
		else
		{
			bytes = 0;
			err = sIOWrite(memIO, sizeof(PGPSize), TRUE, &bytes);
		}
		break;
		
	case kPGPPrefType_Byte:
		if (IsntNull(data.p))
		{
			err = sIOWrite(memIO, sizeof(PGPSize), TRUE, &dataLength);
			
			if (IsntPGPError(err) && dataLength)
				err = sIOWrite(memIO, dataLength, FALSE, data.p);
		}
		else
		{
			bytes = 0;
			err = sIOWrite(memIO, sizeof(PGPSize), TRUE, &bytes);
		}
		break;

	case kPGPPrefType_Struct:
		err = sIOWrite(memIO, sizeof(PGPByte), FALSE, 
				&(data.strct->numMembers));

		if (IsPGPError(err))
			break;

		for (index=0; index<data.strct->numMembers; index++)
		{
			err = sIOWrite(memIO, sizeof(PGPUInt32), TRUE, 
					&(data.strct->members[index].type));

			if (IsPGPError(err))
				break;
			
			switch (data.strct->members[index].type)
			{
			case kPGPPrefType_Boolean:
				strctData.b = (PGPBoolean) data.strct->members[index].data;
				break;
				
			case kPGPPrefType_Number:
				strctData.i = (PGPUInt32) data.strct->members[index].data;
				break;
				
			case kPGPPrefType_String:
				strctData.str = (char *) data.strct->members[index].data;
				break;
				
			case kPGPPrefType_Byte:
				strctData.p = (PGPByte *) data.strct->members[index].data;
				break;
			}

			err = sExportPrefData(memIO, data.strct->members[index].type,
					data.strct->members[index].size, strctData);
		}
		break;
	}

	return err;
}


PGPError PGPExportPrefsToBuffer(PGPPrefRef prefRef, 
									PGPSize *bufferSize,
									void **buffer)
{
	PGPIORef			memIO = NULL;
	PGPFileOffset		fileOffset;
	PGPPrefData *		currentPrefData;
	PGPPrefDataUnion	data;
	void *				arrayData;
	PGPSize				bytesRead;
	PGPUInt32			index;
	PGPError			err = kPGPError_NoErr;
	
	if (IsntNull(bufferSize))
		*bufferSize = 0;
	if (IsntNull(buffer))
		*buffer = NULL;

	PGPValidatePref(prefRef);
	PGPValidatePtr(bufferSize);
	PGPValidatePtr(buffer);

	err = PGPNewMemoryIO(prefRef->memoryMgr, (PGPMemoryIORef *) &memIO);
	
	if (IsntPGPError(err))
	{
		currentPrefData = prefRef->prefList;
		
		while (IsntNull(currentPrefData))
		{
			if (currentPrefData->prefIndex == -1)
			{
				sGetNextPrefData(currentPrefData, &currentPrefData);
				continue;
			}

			err = sIOWrite(memIO, sizeof(PGPUInt32), TRUE,
					&(currentPrefData->prefIndex));

			if (IsPGPError(err))
				break;

			err = sIOWrite(memIO, sizeof(PGPUInt32), TRUE,
					&(currentPrefData->prefType));

			if (IsPGPError(err))
				break;

			err = sIOWrite(memIO, sizeof(PGPUInt32), TRUE,
					&(currentPrefData->flags));

			if (IsPGPError(err))
				break;

			switch (currentPrefData->prefType)
			{
			case kPGPPrefType_Array:
				err = sIOWrite(memIO, sizeof(PGPUInt32), TRUE,
						&(currentPrefData->data.a->numElements));

				if (IsPGPError(err))
					break;

				err = sIOWrite(memIO, sizeof(PGPUInt32), TRUE,
						&(currentPrefData->data.a->type));

				if (IsPGPError(err))
					break;

				if (currentPrefData->data.a->type == kPGPPrefType_Struct)
				{
					data.strct = currentPrefData->data.a->templateStruct;
					PGPValidatePtr(data.strct);

					err = sExportPrefData(memIO, kPGPPrefType_Struct,
							sizeof(PGPPrefStruct), data);

					if (IsPGPError(err))
						break;
				}

				for (index=0; index<currentPrefData->data.a->numElements;
				index++)
				{
					arrayData = currentPrefData->data.a->elements[index].data;
					switch (currentPrefData->data.a->type)
					{
					case kPGPPrefType_Boolean:
						data.b = (PGPBoolean) arrayData;
						break;
					
					case kPGPPrefType_Number:
						data.i = (PGPUInt32) arrayData;
						break;
					
					case kPGPPrefType_String:
						data.str = (char *) arrayData;
						break;
					
					case kPGPPrefType_Byte:
						data.p = (PGPByte *) arrayData;
						break;
					
					case kPGPPrefType_Struct:
						data.strct = (PGPPrefStruct *) arrayData;
						break;
					}

					err = sExportPrefData(memIO, 
							currentPrefData->data.a->type, 
							currentPrefData->data.a->elements[index].size,
							data);
				}
				break;

			default:
				err = sExportPrefData(memIO, currentPrefData->prefType,
						currentPrefData->dataLength, currentPrefData->data);
			}

			if (IsPGPError(err))
				break;

			sGetNextPrefData(currentPrefData, &currentPrefData);
		}
	}

	if (IsntPGPError(err))
		err = PGPIOGetEOF(memIO, &fileOffset);

	if (IsntPGPError(err))
	{
		if (fileOffset > MAX_PGPSize)
			*bufferSize = MAX_PGPSize;
		else
			*bufferSize = (PGPSize) fileOffset;

		*buffer = PGPNewData(prefRef->memoryMgr, *bufferSize, 
					kPGPMemoryMgrFlags_Clear);

		if (IsNull(*buffer))
			err = kPGPError_OutOfMemory;
	}

	if (IsntPGPError(err))
		err = PGPIOSetPos(memIO, 0);

	if (IsntPGPError(err))
		err = PGPIORead(memIO, *bufferSize, *buffer, &bytesRead);

	if (IsPGPError(err))
	{
		*bufferSize = 0;

		if (IsntNull(*buffer))
		{
			PGPFreeData(*buffer);
			*buffer = NULL;
		}
	}

	if (IsntNull(memIO))
		PGPFreeIO(memIO);

	return err;
}


static PGPError sImportPrefData(PGPIORef memIO, PGPPrefType prefType,
									PGPPrefStruct *templateStruct,
									PGPSize *dataLength, void **data)
{
	PGPError		err = kPGPError_NoErr;
	PGPUInt32		index;
	PGPBoolean		crossPlatform;
	PGPBoolean		newStruct;
	PGPByte			numMembers;
	void *			memberData;
	PGPPrefStruct *	prefStruct = NULL;
	PGPPrefType		memberType;
	PGPSize			memberSize;
	PGPMemoryMgrRef	memoryMgr;

	*dataLength = 0;
	*data = NULL;

	memoryMgr = PGPIOGetMemoryMgr(memIO);

	switch (prefType)
	{
	case kPGPPrefType_Boolean:
		*dataLength = sizeof(PGPBoolean);
		crossPlatform = FALSE;
		break;

	case kPGPPrefType_Number:
		*dataLength = sizeof(PGPUInt32);
		crossPlatform = TRUE;
		break;

	case kPGPPrefType_String:
	case kPGPPrefType_Byte:
		err = sIORead(memIO, sizeof(PGPSize), TRUE, dataLength);
		crossPlatform = FALSE;
		break;

	case kPGPPrefType_Struct:
		err = sIORead(memIO, sizeof(PGPByte), FALSE, &numMembers);
		if (IsPGPError(err))
			break;

		if (IsntNull(templateStruct))
		{
			newStruct = FALSE;
			err = sCreatePrefStruct(memoryMgr, templateStruct, &prefStruct);
			if (IsPGPError(err))
				break;
		}
		else
		{
			newStruct = TRUE;
			prefStruct = (PGPPrefStruct *) PGPNewData(memoryMgr,
											sizeof(PGPPrefStruct),
											kPGPMemoryMgrFlags_Clear);
			
			if (IsNull(prefStruct))
			{
				err = kPGPError_OutOfMemory;
				break;
			}
			
			prefStruct->members = (PGPPrefStructMember *) 
								PGPNewData(memoryMgr,
									sizeof(PGPPrefStructMember) * numMembers,
									kPGPMemoryMgrFlags_Clear);
			
			if (IsNull(prefStruct->members))
			{
				err = kPGPError_OutOfMemory;
				break;
			}
			
			prefStruct->numMembers = numMembers;
		}

		for (index=0; index<numMembers; index++)
		{
			err = sIORead(memIO, sizeof(PGPUInt32), TRUE, &memberType);

			if (IsPGPError(err))
				break;

			err = sImportPrefData(memIO, memberType, NULL, &memberSize, 
					&memberData);

			if (IsPGPError(err))
				break;

			if (newStruct)
				prefStruct->members[index].type = memberType;

			if (index >= prefStruct->numMembers)
			{
				if (prefStruct->members[index].type == kPGPPrefType_String)
				{
					prefStruct->members[index].data = PGPNewData(memoryMgr,
														1,
														kPGPMemoryMgrFlags_Clear);

					prefStruct->members[index].size = 1;
				}

				if (IsntNull(memberData))
					PGPFreeData(memberData);
				continue;
			}

			if (memberType != prefStruct->members[index].type)
			{
				if (prefStruct->members[index].type == kPGPPrefType_String)
				{
					prefStruct->members[index].data = PGPNewData(memoryMgr,
														1,
														kPGPMemoryMgrFlags_Clear);

					prefStruct->members[index].size = 1;
				}

				if (IsntNull(memberData))
					PGPFreeData(memberData);
				continue;
			}

			prefStruct->members[index].size = memberSize;

			switch (prefStruct->members[index].type)
			{
			case kPGPPrefType_Boolean:
				prefStruct->members[index].data = (void *)
												*((PGPBoolean *) memberData);
				PGPFreeData(memberData);
				break;

			case kPGPPrefType_Number:
				prefStruct->members[index].data = (void *)
												*((PGPUInt32 *) memberData);
				PGPFreeData(memberData);
				break;

			default:
				prefStruct->members[index].data = memberData;
				break;
			}
		}

		for (index=numMembers; index<prefStruct->numMembers; index++)
		{
			if (prefStruct->members[index].type == kPGPPrefType_String)
			{
				prefStruct->members[index].data = PGPNewData(memoryMgr,
													1,
													kPGPMemoryMgrFlags_Clear);

				prefStruct->members[index].size = 1;
			}
		}

		if (IsntPGPError(err))
			*data = prefStruct;
		break;
	}

	if (IsntPGPError(err))
	{
		if (prefType == kPGPPrefType_Struct)
			*data = prefStruct;
		else if (*dataLength > 0)
		{
			*data = PGPNewData(memoryMgr, *dataLength, 
						kPGPMemoryMgrFlags_Clear);

			if (IsNull(*data))
				err = kPGPError_OutOfMemory;
			else
				err = sIORead(memIO, *dataLength, crossPlatform, *data);
		}
	}

	return err;
}


PGPError PGPImportBufferToMemoryPrefs(PGPMemoryMgrRef memoryMgr,
										void *buffer, 
										PGPSize bufferSize,
										const PGPPrefDefinition *prefs,
										PGPUInt32 numPrefs,
										PGPPrefRef *prefRef)
{
	PGPIORef		memIO = NULL;
	PGPPrefIndex	prefIndex;
	PGPPrefType		prefType;
	PGPUInt32		prefFlags;
	PGPUInt32		arraySize;
	PGPUInt32		index;
	PGPPrefType		arrayType;
	PGPPrefStruct *	tempStruct = NULL;
	PGPPrefStruct *	templateStruct = NULL;
	PGPPrefArray *	prefArray = NULL;
	PGPPrefData *	currentPrefData;
	PGPSize			dataLength;
	void *			data;
	PGPError		err = kPGPError_NoErr;

	if (IsntNull(prefRef))
		*prefRef = NULL;

	PGPValidateParam(bufferSize > 0);
	PGPValidatePtr(buffer);

	err = PGPNewMemoryPrefs(memoryMgr, prefs, numPrefs, prefRef);
	if (IsPGPError(err))
	{
		if (IsntNull(*prefRef))
		{
			sFreePref(*prefRef);
			*prefRef = NULL;
		}
		
		return err;
	}

	err = PGPNewMemoryIO(memoryMgr, (PGPMemoryIORef *) &memIO);
	
	if (IsntPGPError(err))
		err = PGPIOWrite(memIO, bufferSize, buffer);

	if (IsntPGPError(err))
		err = PGPIOSetPos(memIO, 0);

	while (IsntPGPError(err) && !PGPIOIsAtEOF(memIO))
	{
		err = sIORead(memIO, sizeof(PGPUInt32), TRUE, &prefIndex);
		if (IsPGPError(err))
			break;

		err = sIORead(memIO, sizeof(PGPUInt32), TRUE, &prefType);
		if (IsPGPError(err))
			break;

		err = sIORead(memIO, sizeof(PGPUInt32), TRUE, &prefFlags);
		if (IsPGPError(err))
			break;

		switch (prefType)
		{
		case kPGPPrefType_Array:
			prefArray = NULL;
			tempStruct = NULL;
			templateStruct = NULL;

			err = sIORead(memIO, sizeof(PGPUInt32), TRUE, &arraySize);
			if (IsPGPError(err))
				break;

			err = sIORead(memIO, sizeof(PGPUInt32), TRUE, &arrayType);
			if (IsPGPError(err))
				break;

			if (arrayType == kPGPPrefType_Struct)
			{
				err = sImportPrefData(memIO, arrayType, NULL, &dataLength, 
						&tempStruct);

				if (IsntPGPError(err))
					err = sFindPrefDataByIndex(*prefRef, prefIndex, 
							&currentPrefData);

				if (IsntPGPError(err) && IsntNull(currentPrefData))
					templateStruct = currentPrefData->data.a->templateStruct;
				else
					templateStruct = tempStruct;

				if (IsntPGPError(err))
					err = PGPCreatePrefArray(*prefRef, arrayType, arraySize,
							templateStruct, &prefArray);
			}
			else
				err = PGPCreatePrefArray(*prefRef, arrayType, arraySize, NULL,
						&prefArray);

			if (IsPGPError(err))
				break;

			prefArray->type = arrayType;
			prefArray->numElements = arraySize;

			for (index=0; index<arraySize; index++)
			{
				err = sImportPrefData(memIO, arrayType, templateStruct, 
						&(prefArray->elements[index].size), &data);

				switch (arrayType)
				{
				case kPGPPrefType_Boolean:
					prefArray->elements[index].data = (void *) 
														*((PGPBoolean *) data);
					PGPFreeData(data);
					break;

				case kPGPPrefType_Number:
					prefArray->elements[index].data = (void *)
														*((PGPUInt32 *) data);
					PGPFreeData(data);
					break;

				default:
					prefArray->elements[index].data = data;
					break;
				}

				if (IsPGPError(err))
					break;
			}

			if (IsntPGPError(err))
				err = PGPSetPrefData(*prefRef, prefIndex, sizeof(PGPPrefArray),
						prefArray);

			if (IsntNull(prefArray))
				PGPFreePrefArray(prefArray);
			if (IsntNull(tempStruct))
				PGPFreePrefStruct(tempStruct);
			break;

		case kPGPPrefType_Struct:
			err = sFindPrefDataByIndex(*prefRef, prefIndex, &currentPrefData);
			
			if (IsntPGPError(err) && IsntNull(currentPrefData))
				templateStruct = currentPrefData->data.a->templateStruct;
			else
				templateStruct = NULL;

			err = sImportPrefData(memIO, prefType, templateStruct, 
					&dataLength, &data);

			if (IsntPGPError(err))
				err = PGPSetPrefData(*prefRef, prefIndex, dataLength, data);

			if (IsntNull(data))
				PGPFreePrefStruct(data);
			break;

		default:
			err = sImportPrefData(memIO, prefType, NULL, &dataLength, &data);
			if (IsntPGPError(err))
				err = PGPSetPrefData(*prefRef, prefIndex, dataLength, data);

			if (IsntNull(data))
					PGPFreeData(data);
			break;
		}

		if (IsntPGPError(err))
			err = sCopyPrefFlags(*prefRef, prefIndex, prefFlags);

		if (err == kPGPError_PrefNotFound)
			err = kPGPError_NoErr;
	}

	if (IsntNull(memIO))
		PGPFreeIO(memIO);

	if (IsPGPError(err))
	{
		if (IsntNull(*prefRef))
		{
			sFreePref(*prefRef);
			*prefRef = NULL;
		}
	}

	return err;
}


PGPError PGPGetPrefFlags(PGPPrefRef prefRef,
							PGPPrefIndex prefIndex,
							PGPUInt32 *flags)
{
	PGPPrefData *	currentPrefData;
	PGPError		err = kPGPError_NoErr;

	if (IsntNull(flags))
		*flags = 0;

	PGPValidatePref(prefRef);
	PGPValidatePtr(flags);

	err = sFindPrefDataByIndex(prefRef, prefIndex, &currentPrefData);
	if (IsPGPError(err))
		return err;

	if (IsNull(currentPrefData))
		return kPGPError_PrefNotFound;

	*flags = currentPrefData->flags;
	return err;
}


PGPError PGPSetPrefFlags(PGPPrefRef prefRef,
							PGPPrefIndex prefIndex,
							PGPUInt32 bitmask)
{
	PGPPrefData *	currentPrefData;
	PGPUInt32		oldFlags;
	PGPError		err = kPGPError_NoErr;

	PGPValidatePref(prefRef);

	err = sFindPrefDataByIndex(prefRef, prefIndex, &currentPrefData);
	if (IsPGPError(err))
		return err;

	if (IsNull(currentPrefData))
		return kPGPError_PrefNotFound;

	oldFlags = currentPrefData->flags;
	currentPrefData->flags |= bitmask;
	if (currentPrefData->flags != oldFlags)
		prefRef->dirty = TRUE;

	return err;
}


PGPError PGPClearPrefFlags(PGPPrefRef prefRef,
							PGPPrefIndex prefIndex,
							PGPUInt32 bitmask)
{
	PGPPrefData *	currentPrefData;
	PGPUInt32		oldFlags;
	PGPError		err = kPGPError_NoErr;

	PGPValidatePref(prefRef);

	err = sFindPrefDataByIndex(prefRef, prefIndex, &currentPrefData);
	if (IsPGPError(err))
		return err;

	if (IsNull(currentPrefData))
		return kPGPError_PrefNotFound;

	oldFlags = currentPrefData->flags;
	currentPrefData->flags &= ~bitmask;
	if (currentPrefData->flags != oldFlags)
		prefRef->dirty = TRUE;

	return err;
}


static PGPError sCopyPrefFlags(PGPPrefRef prefRef,
					PGPPrefIndex prefIndex,
					PGPUInt32 flags)
{
	PGPPrefData *	currentPrefData;
	PGPError		err = kPGPError_NoErr;

	PGPValidatePref(prefRef);

	err = sFindPrefDataByIndex(prefRef, prefIndex, &currentPrefData);
	if (IsPGPError(err))
		return err;

	if (IsNull(currentPrefData))
		return kPGPError_PrefNotFound;

	if (currentPrefData->flags != flags)
	{
		prefRef->dirty = TRUE;
		currentPrefData->flags = flags;
	}

	return err;
}


static PGPError sFindPrefDataByIndex(PGPPrefRef prefRef, 
					PGPPrefIndex prefIndex, 
					PGPPrefData **prefData)
{
	PGPPrefData *searchPrefData;
	
	if (IsntNull(prefData))
		*prefData = NULL;

	PGPValidatePtr(prefData);
	PGPValidatePref(prefRef);

	searchPrefData = prefRef->prefList;
	while (IsntNull(searchPrefData))
	{
		if (searchPrefData->prefIndex == prefIndex)
			break;
		
		sGetNextPrefData(searchPrefData, &searchPrefData);
	}

	*prefData = searchPrefData;
	return kPGPError_NoErr;
}


static PGPError sFindPrefDataByName(PGPPrefRef prefRef,
					char *prefName, 
					PGPPrefData **prefData)
{
	PGPPrefData *searchPrefData;

	if (IsntNull(prefData))
		*prefData = NULL;

	PGPValidatePtr(prefData);
	PGPValidatePtr(prefName);
	PGPValidatePref(prefRef);

	searchPrefData = prefRef->prefList;
	while (IsntNull(searchPrefData))
	{
		if (!pgpCompareStringsIgnoreCase(searchPrefData->prefName, prefName))
			break;

		sGetNextPrefData(searchPrefData, &searchPrefData);
	}

	*prefData = searchPrefData;
	return kPGPError_NoErr;
}


static PGPError sFreePref(PGPPrefRef prefRef)
{
	PGPPrefData *currentPrefData;
	PGPPrefData *tempPrefData;
	PGPMemoryMgrRef memoryMgr;

	PGPValidatePref(prefRef);

	currentPrefData = prefRef->prefList;
	while (IsntNull(currentPrefData))
	{
		sGetNextPrefData(currentPrefData, &tempPrefData);
		sRemovePrefData(prefRef, currentPrefData);
		currentPrefData = tempPrefData;
	}

	if (pflFileSpecIsValid(prefRef->prefFileSpec))
		PFLFreeFileSpec(prefRef->prefFileSpec);

	memoryMgr = prefRef->memoryMgr;
	PGPFreeData( prefRef);
	return kPGPError_NoErr;
}


static PGPError sAddPrefData(PGPPrefRef prefRef, 
					PGPPrefData *prefData)
{
	PGPValidatePref(prefRef);
	PGPValidatePtr(prefData);

	if (IsNull(prefRef->prefList))
		prefRef->prefList = prefData;

	if (IsntNull(prefRef->prefListEnd))
		prefRef->prefListEnd->nextPrefData = prefData;

	prefData->lastPrefData	= prefRef->prefListEnd;
	prefData->nextPrefData	= NULL;
	prefRef->prefListEnd	= prefData;
	return kPGPError_NoErr;
}


static PGPError sRemovePrefData(PGPPrefRef prefRef, 
					PGPPrefData *prefData)
{
	PGPValidatePref(prefRef);
	PGPValidatePtr(prefData);

	sFreePrefData(prefData, TRUE);

	if (IsntNull(prefData->nextPrefData))
		prefData->nextPrefData->lastPrefData = prefData->lastPrefData;

	if (IsntNull(prefData->lastPrefData))
		prefData->lastPrefData->nextPrefData = prefData->nextPrefData;

	if (prefRef->prefList == prefData)
		prefRef->prefList = prefData->nextPrefData;
	
	if (prefRef->prefListEnd == prefData)
		prefRef->prefListEnd = prefData->lastPrefData;

	PGPFreeData(prefData);
	return kPGPError_NoErr;
}


static PGPError sGetNextPrefData(const PGPPrefData *currentPrefData,
					PGPPrefData **nextPrefData)
{
	if (IsntNull(nextPrefData))
		*nextPrefData = NULL;

	PGPValidatePtr(nextPrefData);
	PGPValidatePtr(currentPrefData);

	*nextPrefData = currentPrefData->nextPrefData;
	return kPGPError_NoErr;
}


	static PGPError
sReadPrefData(
	PGPIORef			prefIO,
	PGPPrefData *		prefData,
	PGPUInt32 *			arrayIndex )
{
	PGPMemoryMgrRef		memoryMgr;
	PGPSize				bytesRead;
	char *				lineBuffer;
	char *				equalSign;
	char *				flagMarker;
	char *				hexMarker;
	char *				value;
	char *				indexNum;
	PGPBoolean			gotValue;
	PGPBoolean			openQuote;
	PGPUInt32			lineSize = 4096;
	PGPUInt32			index;
	PGPUInt32			commentsSize;
	PGPFileOffset		position;
	PGPError			err = kPGPError_NoErr;

	memoryMgr = PGPIOGetMemoryMgr(prefIO);

	lineBuffer = (char *) PGPNewData( memoryMgr, lineSize,
							kPGPMemoryMgrFlags_Clear);
	if( IsNull( lineBuffer ) )
	{
		err = kPGPError_OutOfMemory;
		goto done;
	}
	commentsSize = 0;
	index = 0;
	gotValue = FALSE;
	prefData->prefIndex = kPGPInvalidPrefIndex;
	prefData->prefType = kPGPPrefType_Invalid;
	*arrayIndex = 0;

	do
	{
		do
		{
			while( index + 1024 > lineSize )
			{
				lineSize += 1024;
				err = PGPReallocData( memoryMgr, &lineBuffer, lineSize,
						kPGPMemoryMgrFlags_Clear );
				if( IsPGPError( err ) )
					goto done;
			}
			err = PGPIORead(prefIO, 1, &(lineBuffer[index]), &bytesRead);

			if ((index > 0) && IsLineEndChar(lineBuffer[index]))
			{
				while (IsWhitespace(lineBuffer[index-1]) && (index > 2))
				{
					index--;
					lineBuffer[index] = lineBuffer[index+1];
				}

				if ((lineBuffer[index-1] == '\\') && (lineBuffer[0] != '#'))
				{
					index--;

					while (index > 0)
					{
						if (!IsWhitespace(lineBuffer[index-1]))
							break;

						index--;
					}

					do
					{
						err = PGPIORead(prefIO, 1, &(lineBuffer[index]),
								&bytesRead);
					}
					while (IsntPGPError(err) && 
							IsLineEndChar(lineBuffer[index]));

					if (IsntPGPError(err))
					{
						if ((lineBuffer[index] == '\"') &&
							(lineBuffer[index-1] == '\"'))
						{
							index--;
							err = PGPIORead(prefIO, 1, &(lineBuffer[index]),
									&bytesRead);
						}
					}

				}
			}
			
			index++;
		}
		while (IsntPGPError(err) && !IsLineEndChar(lineBuffer[index-1]) &&
				(index < lineSize));

		index--;
		do
		{
			err = PGPIORead(prefIO, 1, &(lineBuffer[index]), &bytesRead);
		}
		while (IsntPGPError(err) && IsLineEndChar(lineBuffer[index]));

		if (err == kPGPError_EOF)
			err = kPGPError_NoErr;
		else if (IsPGPError(err))
			break;
		else
		{
			PGPIOGetPos(prefIO, &position);
			position--;
			PGPIOSetPos(prefIO, position);
		}

		lineBuffer[index] = 0;

		switch (lineBuffer[0])
		{
		case '#':
			err = PGPReallocData(memoryMgr, &(prefData->prefComments),
					commentsSize + index + strlen(kPGPLineEndString) + 1, 0);
			
			if (IsntPGPError(err))
			{
				pgpCopyMemory(lineBuffer, 
					&(prefData->prefComments[commentsSize]),
					index);
				pgpCopyMemory(kPGPLineEndString, 
					&(prefData->prefComments[commentsSize+index]),
					strlen(kPGPLineEndString)+1);
				commentsSize += index + strlen(kPGPLineEndString);
			}
			break;

		case ' ':
		case '\r':
		case '\n':
			break;

		default:
			equalSign = strchr(lineBuffer, '=');
			
			if (IsntNull(equalSign))
			{
				indexNum = equalSign - 1;
				while ((*indexNum >= '0') && (*indexNum <= '9'))
					indexNum--;

				indexNum++;
				if (indexNum != equalSign)
					sscanf(indexNum, "%d", arrayIndex);

				prefData->prefName = (char *) PGPNewData(memoryMgr,
												equalSign - lineBuffer + 1,
												kPGPMemoryMgrFlags_Clear);

				if (IsntNull(prefData->prefName))
				{
					pgpCopyMemory(lineBuffer, prefData->prefName,
						indexNum - lineBuffer);
				}
				else
					err = kPGPError_OutOfMemory;

				if (IsntPGPError(err))
				{
					flagMarker = &(equalSign[strlen(equalSign) - 1]);
					openQuote = FALSE;
					while (flagMarker > equalSign)
					{
						if (*flagMarker == '\"')
							openQuote = !openQuote;

						if ((*flagMarker == ';') && !openQuote)
							break;

						flagMarker--;
					}

					if (flagMarker == equalSign)
						flagMarker = &(lineBuffer[index]);

					do
						flagMarker--;
					while (IsWhitespace(*flagMarker));
					
					flagMarker++;

					if (index != (PGPUInt32)(equalSign - lineBuffer + 1))
					{
						value = (char *) PGPNewData(memoryMgr,
											flagMarker - equalSign,
											kPGPMemoryMgrFlags_Clear);

						if (IsntNull(value))
							pgpCopyMemory(equalSign+1, value,
								flagMarker - equalSign - 1);
						else
							err = kPGPError_OutOfMemory;
					}
					else
						value = NULL;
				}
				
				if (IsntPGPError(err))
				{
					prefData->flags = 0;
					
					if (flagMarker != &(lineBuffer[index]))
					{
						hexMarker = strchr(flagMarker, 'x');
						if (IsNull(hexMarker))
							hexMarker = strchr(flagMarker, 'X');

						if (IsntNull(hexMarker))
							prefData->flags = pgpHexToPGPUInt32(hexMarker+1);
						else
						{
							flagMarker += strcspn(flagMarker, "01234567890");
							sscanf(flagMarker, "%d", &(prefData->flags));
						}
					}

					if (IsntNull(value))
						prefData->dataLength = strlen(value) + 1;
					else
						prefData->dataLength = 0;

					prefData->data.raw = value;
					gotValue = TRUE;
				}
			}
		}

		index = 0;
	}
	while (IsntPGPError(err) && !PGPIOIsAtEOF(prefIO) && !gotValue);
done:
	if( IsntNull( lineBuffer ) )
		(void) PGPFreeData( lineBuffer );
	return err;
}


	static PGPError
sWritePrefData(
	PGPIORef		prefIO,
	PGPPrefData		*prefData)
{
	PGPMemoryMgrRef memoryMgr;
	PGPUInt32		begin;
	PGPUInt32		end;
	PGPUInt32		index;
	PGPError		err = kPGPError_NoErr;

	memoryMgr = PGPIOGetMemoryMgr(prefIO);

	begin = 0;
	end = 0;

	if (IsntNull(prefData->prefComments))
	{
		err = PGPIOWrite(prefIO, strlen(kPGPLineEndString), 
				kPGPLineEndString);

		while (IsntPGPError(err) && (begin < strlen(prefData->prefComments)))
		{
			while (!IsLineEndChar(prefData->prefComments[end]) &&
					(prefData->prefComments[end] != '\0'))
				end++;

			err = PGPIOWrite(prefIO, end - begin, 
					&(prefData->prefComments[begin]));

			if (IsPGPError(err))
				break;

			while (IsLineEndChar(prefData->prefComments[end]) &&
					(prefData->prefComments[end] != '\0'))
				end++;

			begin = end;

			err = PGPIOWrite(prefIO, strlen(kPGPLineEndString),
					kPGPLineEndString);
		}
	}

	if (prefData->prefType == kPGPPrefType_Array)
	{
		for (index=0; index<prefData->data.a->numElements; index++)
		{
			err = sWritePrefLine(prefIO, prefData, index);

			if (IsntPGPError(err))
				err = PGPIOWrite(prefIO, strlen(kPGPLineEndString), 
						kPGPLineEndString);

			if (IsPGPError(err))
				break;
		}
	}
	else
	{
		err = sWritePrefLine(prefIO, prefData, 0);
	
		if (IsntPGPError(err))
			err = PGPIOWrite(prefIO, strlen(kPGPLineEndString), 
					kPGPLineEndString);
	}

	return err;
}


	static PGPError
sWritePrefLine(
	PGPIORef			prefIO,
	PGPPrefData *		prefData,
	PGPUInt32			index )
{
	PGPMemoryMgrRef		memoryMgr;
	PGPUInt32			begin;
	PGPUInt32			end;
	PGPUInt32			length;
	PGPUInt32			count;
	PGPUInt32			comma;
	PGPUInt32			formattedSize = 4096;
	PGPUInt32			hexSize = 4096;
	char *				formattedLine = NULL;
	char *				hexString = NULL;
	char				number[20];
	PGPBoolean			openQuote;
	PGPPrefDataUnion	data;
	PGPPrefType			type;
	PGPSize				dataLength;
	PGPPrefStructMember	*member;
	PGPError			err = kPGPError_NoErr;

	memoryMgr = PGPIOGetMemoryMgr(prefIO);

	hexString = (char *) PGPNewData(memoryMgr, hexSize,
								kPGPMemoryMgrFlags_Clear);
	formattedLine = (char *) PGPNewData(memoryMgr, formattedSize,
								kPGPMemoryMgrFlags_Clear);
	if( IsNull( hexString ) || IsNull( formattedLine ) )
	{
		err = kPGPError_OutOfMemory;
		goto done;
	}
	if (IsntNull(prefData->prefName))
	{
		if (prefData->prefType == kPGPPrefType_Array)
			sprintf(formattedLine, "%s%d=", prefData->prefName, index+1);
		else
			sprintf(formattedLine, "%s=", prefData->prefName);
	}
	else
		goto done;

	err = PGPIOWrite(prefIO, strlen(formattedLine), formattedLine);

	if (prefData->prefType == kPGPPrefType_Array)
	{
		type = prefData->data.a->type;
		dataLength = prefData->data.a->elements[index].size;

		switch (type)
		{
		case kPGPPrefType_Boolean:
			data.b = (PGPBoolean) prefData->data.a->elements[index].data;
			break;

		case kPGPPrefType_Number:
			data.i = (PGPUInt32) prefData->data.a->elements[index].data;
			break;

		case kPGPPrefType_String:
			data.str = (char *) prefData->data.a->elements[index].data;
			break;

		case kPGPPrefType_Byte:
			data.p = (PGPByte *) prefData->data.a->elements[index].data;
			break;

		case kPGPPrefType_Struct:
			data.strct = (PGPPrefStruct *) 
							prefData->data.a->elements[index].data;
			break;
		}
	}
	else
	{
		data = prefData->data;
		type = prefData->prefType;
		dataLength = prefData->dataLength;
	}

	if (IsntPGPError(err))
	{
		switch (type)
		{
		case kPGPPrefType_Boolean:
			if (data.b)
				strcpy(formattedLine, "TRUE");
			else
				strcpy(formattedLine, "FALSE");
			break;

		case kPGPPrefType_Number:
			sprintf(formattedLine, "%d", data.i);
			break;

		case kPGPPrefType_String:
			sprintf(formattedLine, "\"%s\"", data.str);
			break;

		case kPGPPrefType_Byte:
			if (IsntNull(data.p) && dataLength)
			{
				if ( dataLength * 2 + 8 > hexSize )
				{
					hexSize = dataLength * 2 + 8;
					err = PGPReallocData( memoryMgr, &hexString, hexSize,
						kPGPMemoryMgrFlags_Clear );
					if( IsPGPError( err ) )
						goto done;
				}
				pgpBytesToHex(data.p, dataLength, TRUE, hexString);
				if ( strlen(hexString) + strlen(formattedLine) + 1 >
					formattedSize )
				{
					formattedSize = strlen(hexString) + strlen(formattedLine) + 1;
					err = PGPReallocData( memoryMgr, &formattedLine,
						formattedSize, kPGPMemoryMgrFlags_Clear );
					if( IsPGPError( err ) )
						goto done;
				}
				strcpy(formattedLine, hexString);
			}
			else
				formattedLine[0] = 0;
			break;

		case kPGPPrefType_Struct:
			formattedLine[0] = 0;

			for (count=0; count<data.strct->numMembers; count++)
			{
				if (count > 0)
					strcat(formattedLine, ",");

				member = &(data.strct->members[count]);

				switch (member->type)
				{
				case kPGPPrefType_Boolean:
					if (member->data)
						strcat(formattedLine, "TRUE");
					else
						strcat(formattedLine, "FALSE");
					break;

				case kPGPPrefType_Number:
					sprintf(number, "%d", (PGPUInt32) member->data);
					strcat(formattedLine, number);
					break;

				case kPGPPrefType_String:
					strcat(formattedLine, "\"");
					strcat(formattedLine, (char *) member->data);
					strcat(formattedLine, "\"");
					break;

				case kPGPPrefType_Byte:
					if (IsntNull(member->data) && member->size)
					{
						if ( member->size * 2 + 8 > hexSize )
						{
							hexSize = member->size * 2 + 8;
							err = PGPReallocData( memoryMgr, &hexString, hexSize,
								kPGPMemoryMgrFlags_Clear );
							if( IsPGPError( err ) )
								goto done;
						}
						pgpBytesToHex(member->data, member->size, TRUE, 
							hexString);
	
						if ( strlen(hexString) + strlen(formattedLine) + 1 >
							formattedSize )
						{
							formattedSize = strlen(hexString) +
											strlen(formattedLine) + 128;
							err = PGPReallocData( memoryMgr, &formattedLine,
								formattedSize, kPGPMemoryMgrFlags_Clear );
							if( IsPGPError( err ) )
								goto done;
						}
						strcat(formattedLine, hexString);
					}
					break;
				}
			}
			break;

		case kPGPPrefType_Invalid:
			if (IsntNull(data.raw))
			{
				if ( dataLength + 1 > formattedSize )
				{
					formattedSize = dataLength + 1;
					err = PGPReallocData( memoryMgr, &formattedLine,
						formattedSize, kPGPMemoryMgrFlags_Clear );
					if( IsPGPError( err ) )
						goto done;
				}
				strcpy(formattedLine, data.raw);
			}
			else
				formattedLine[0] = 0;
			break;
		}

		if (prefData->flags)
		{
			strcat(formattedLine, " ;");
			pgpPGPUInt32ToHex(prefData->flags, TRUE, hexString);
			strcat(formattedLine, hexString);
		}

		begin = 0;
		end = 0;
		length = strlen(formattedLine);
		openQuote = FALSE;

		while (IsntPGPError(err) && (begin < length))
		{
			end += 70;

			if (end > length)
				end = length;

			comma = end;
			for (count=begin; count<end; count++)
			{
				if (formattedLine[count] == '\"')
					openQuote = !openQuote;

				if (!openQuote && (formattedLine[count] == ','))
					comma = count;
			}

			if (!openQuote && (end < length))
				end = comma + 1;

			err = PGPIOWrite(prefIO, end - begin, &(formattedLine[begin]));

			begin = end;
			if (begin < length)
			{
				if (openQuote)
					err = PGPIOWrite(prefIO, 1, "\"");

				if (IsntPGPError(err))
					err = PGPIOWrite(prefIO, 2, " \\");

				if (IsntPGPError(err))
					err = PGPIOWrite(prefIO, strlen(kPGPLineEndString),
							kPGPLineEndString);

				if (IsntPGPError(err) && openQuote)
					err = PGPIOWrite(prefIO, 1, "\"");
			}
		}
	}
done:
	if( IsntNull( formattedLine ) )
		(void) PGPFreeData( formattedLine );
	if( IsntNull( hexString ) )
		(void) PGPFreeData( hexString );
	return err;
}


static void sFreePrefData(PGPPrefData *prefData, PGPBoolean freeAll)
{
	if (IsntNull(prefData->prefComments) && freeAll)
	{
		PGPFreeData(prefData->prefComments);
		prefData->prefComments = NULL;
	}

	if (IsntNull(prefData->prefName) && freeAll)
	{
		PGPFreeData(prefData->prefName);
		prefData->prefName = NULL;
	}

	switch (prefData->prefType)
	{
	case kPGPPrefType_String:
		if (IsntNull(prefData->data.str))
		{
			PGPFreeData(prefData->data.str);
			prefData->data.str = NULL;
		}
		break;

	case kPGPPrefType_Byte:
		if (IsntNull(prefData->data.p))
		{
			PGPFreeData(prefData->data.p);
			prefData->data.p = NULL;
		}
		break;

	case kPGPPrefType_Struct:
		if (IsntNull(prefData->data.strct))
		{
			PGPFreePrefStruct(prefData->data.strct);
			prefData->data.strct = NULL;
		}
		break;

	case kPGPPrefType_Array:
		if (IsntNull(prefData->data.a))
		{
			PGPFreePrefArray(prefData->data.a);
			prefData->data.a = NULL;
		}
		break;

	case kPGPPrefType_Invalid:
		if (IsntNull(prefData->data.raw))
		{
			PGPFreeData(prefData->data.raw);
			prefData->data.raw = NULL;
		}
		break;
	}

	return;
}


static PGPError sConvertRawData(PGPMemoryMgrRef memoryMgr, 
					PGPPrefData *oldPrefData, PGPPrefData *newPrefData,
					PGPUInt32 arrayIndex)
{
	char *value;
	PGPPrefArrayElement *element;
	PGPPrefStruct *prefStruct;
	PGPUInt32 length;
	PGPUInt32 index;
	PGPUInt32 structIndex;
	PGPError err = kPGPError_NoErr;

	value = newPrefData->data.raw;

	if (IsntNull(value))
		length = strlen(value);
	else
		length = 0;

	if ((oldPrefData->prefType != kPGPPrefType_Array) || (arrayIndex == 1))
	{
		if (IsntNull(oldPrefData->prefComments))
			PGPFreeData(oldPrefData->prefComments);

		oldPrefData->prefComments = newPrefData->prefComments;
		oldPrefData->flags = newPrefData->flags;
	}
	else
	{
		if (IsntNull(newPrefData->prefComments))
			PGPFreeData(newPrefData->prefComments);

		newPrefData->prefComments = NULL;
	}

	switch (oldPrefData->prefType)
	{
	case kPGPPrefType_Boolean:
		oldPrefData->dataLength = sizeof(PGPBoolean);
		if (IsntNull(value))
		{
			oldPrefData->data.b = !pgpCompareStringsIgnoreCase(value, "true");
			PGPFreeData(value);
		}
		else
			oldPrefData->data.b = 0;
		break;
		
	case kPGPPrefType_Number:
		oldPrefData->dataLength = sizeof(PGPUInt32);
		if (IsntNull(value))
		{
			sscanf(value, "%d", &(oldPrefData->data.i));
			PGPFreeData(value);
		}
		else
			oldPrefData->data.i = 0;
		break;
		
	case kPGPPrefType_String:
		if (IsntNull(oldPrefData->data.str))
			PGPFreeData(oldPrefData->data.str);

		if (IsntNull(value))
		{
			sStripQuotes(length + 1, value);
			oldPrefData->dataLength = strlen(value) + 1;
		}
		else
			oldPrefData->dataLength = 0;

		oldPrefData->data.str = value;
		break;
		
	case kPGPPrefType_Byte:
		if (IsntNull(oldPrefData->data.p))
			PGPFreeData(oldPrefData->data.p);

		if (IsntNull(value) && ( length >= 2 ) )
		{
			oldPrefData->dataLength = (length - 2) / 2;
			oldPrefData->data.p = PGPNewData(memoryMgr,
								oldPrefData->dataLength,
								kPGPMemoryMgrFlags_Clear);
		
			if (IsntNull(oldPrefData->data.p))
				pgpHexToBytes(value, oldPrefData->dataLength,
					oldPrefData->data.p);
			else
				err = kPGPError_OutOfMemory;
		
			PGPFreeData(value);
		}
		else
		{
			oldPrefData->dataLength = 0;
			oldPrefData->data.p = NULL;
			
			if( IsntNull( value ) )
				PGPFreeData( value );
		}
		break;

	case kPGPPrefType_Struct:
		if (IsntNull(value))
		{
			err = sConvertRawDataToStruct(memoryMgr, value, length, 
					oldPrefData->data.strct);

			PGPFreeData(value);
		}
		break;

	case kPGPPrefType_Array:
		if (arrayIndex < oldPrefData->data.a->numElements)
		{
			switch (oldPrefData->data.a->type)
			{
			case kPGPPrefType_String:
			case kPGPPrefType_Byte:
				for (index=arrayIndex; 
				index<oldPrefData->data.a->numElements; index++)
				{
					if (IsntNull(oldPrefData->data.a->elements[index].data))
					{
						PGPFreeData(oldPrefData->data.a->elements[index].data);
						oldPrefData->data.a->elements[index].data = NULL;
					}
				}
				break;
				
			case kPGPPrefType_Struct:
				for (index=arrayIndex; 
				index<oldPrefData->data.a->numElements; index++)
				{
					prefStruct = (PGPPrefStruct *) 
								oldPrefData->data.a->elements[index].data;
					
					for (structIndex=0; structIndex<prefStruct->numMembers; 
					structIndex++)
					{
						switch (prefStruct->members[structIndex].type)
						{
						case kPGPPrefType_String:
						case kPGPPrefType_Byte:
							if (IsntNull(prefStruct->members[structIndex].data))
								PGPFreeData(prefStruct->members[structIndex].data);
							break;
						}
					}
					
					PGPFreeData(prefStruct->members);
					prefStruct->members = NULL;
					PGPFreeData(prefStruct);
					oldPrefData->data.a->elements[index].data = NULL;
				}
				break;
			}

			oldPrefData->data.a->numElements = arrayIndex;
		}

		if (arrayIndex > oldPrefData->data.a->numElements)
		{
			err = PGPReallocData(memoryMgr, &(oldPrefData->data.a->elements),
					sizeof(PGPPrefArrayElement) * arrayIndex, 0);

			if (IsPGPError(err))
			{
				if (IsntNull(value))
					PGPFreeData(value);
				break;
			}

			for (index=oldPrefData->data.a->numElements; index<arrayIndex;
			index++)
			{
				oldPrefData->data.a->elements[index].data = NULL;
				oldPrefData->data.a->elements[index].size = 0;
			}

			oldPrefData->data.a->numElements = arrayIndex;
		}

		element = &(oldPrefData->data.a->elements[arrayIndex-1]);

		switch (oldPrefData->data.a->type)
		{
		case kPGPPrefType_Boolean:
			if (IsntNull(value))
			{
				element->data = (void *) !pgpCompareStringsIgnoreCase(value, "true");
				PGPFreeData(value);
			}
			else
				element->data = 0;
			break;

		case kPGPPrefType_Number:
			if (IsntNull(value))
			{
				sscanf(value, "%d", &(element->data));
				PGPFreeData(value);
			}
			else
				element->data = 0;
			break;

		case kPGPPrefType_String:
			if (IsntNull(element->data))
				PGPFreeData(element->data);

			if (IsntNull(value))
			{
				sStripQuotes(length + 1, value);
				element->size = strlen(value) + 1;
			}
			else
				element->size = 0;

			element->data = value;
			break;

		case kPGPPrefType_Byte:
			if (IsntNull(element->data))
				PGPFreeData(element->data);

			if (IsntNull(value))
			{
				element->size = (length - 2) / 2;
				element->data = PGPNewData(memoryMgr,
										element->size,
										kPGPMemoryMgrFlags_Clear);
			
				if (IsntNull(element->data))
					pgpHexToBytes(value, element->size, element->data);
				else
					err = kPGPError_OutOfMemory;
			
				PGPFreeData(value);
			}
			else
			{
				element->size = 0;
				element->data = NULL;
			}
			break;

		case kPGPPrefType_Struct:
			if (IsntNull(value))
			{
				if (IsNull(element->data))
				{
					err = sCreatePrefStruct(memoryMgr,
							oldPrefData->data.a->templateStruct,
							(PGPPrefStruct **) &element->data);

					if (IsPGPError(err))
						break;
				}

				err = sConvertRawDataToStruct(memoryMgr, value, length,
						(PGPPrefStruct *) element->data);

				PGPFreeData(value);
				element->size = sizeof(PGPPrefStruct);
			}
			else
				element->size = 0;
			break;
		}
	}

	newPrefData->data.raw = NULL;
	return err;
}


static PGPError sConvertRawDataToStruct(PGPMemoryMgrRef memoryMgr, 
					char *value, PGPUInt32 length, PGPPrefStruct *prefStruct)
{
	PGPUInt32 begin;
	PGPUInt32 end;
	PGPUInt32 index;
	PGPBoolean openQuote;
	PGPPrefStructMember *member;
	PGPError err = kPGPError_NoErr;

	begin = 0;
	end = 0;
	
	for (index=0; index<prefStruct->numMembers; index++)
	{
		openQuote = FALSE;
		while (((value[end] != ',') || openQuote) && (end < length))
		{
			if (value[end] == '\"')
				openQuote = !openQuote;
			
			end++;
		}
		
		member = &(prefStruct->members[index]);
		
		if (begin < length)
		{
			value[end] = 0;
			sStripQuotes(end - begin + 1, &(value[begin]));
			
			switch (member->type)
			{
			case kPGPPrefType_Boolean:
				member->data = (void *) !pgpCompareStringsIgnoreCase(&(value[begin]), "true");
				break;
				
			case kPGPPrefType_Number:
				sscanf(&(value[begin]), "%d", &(member->data));
				break;
				
			case kPGPPrefType_String:
				if (IsntNull(member->data))
					PGPFreeData(member->data);
				
				member->size = end - begin + 1;
				member->data = PGPNewData(memoryMgr,
									member->size,
									kPGPMemoryMgrFlags_Clear);
				
				if (IsNull(member->data))
					err = kPGPError_OutOfMemory;
				else
					pgpCopyMemory(&(value[begin]), member->data, 
						member->size);
				break;
				
			case kPGPPrefType_Byte:
				if (IsntNull(member->data))
					PGPFreeData(member->data);
				
				if (end > (begin+3))
				{
					member->size = (end - begin - 2) / 2;
					member->data = PGPNewData(memoryMgr,
										member->size,
										kPGPMemoryMgrFlags_Clear);
					
					if (IsNull(member->data))
						err = kPGPError_OutOfMemory;
					else
						pgpHexToBytes(&(value[begin]), member->size, 
							member->data);
				}
				else
					member->data = NULL;

				break;
			}
			
			if (end < length)
				end++;

			begin = end;
		}
		else
		{
			switch (member->type)
			{
			case kPGPPrefType_String:
				if (IsntNull(member->data))
					PGPFreeData(member->data);

				member->data = PGPNewData(memoryMgr, 1, 
								kPGPMemoryMgrFlags_Clear);
				member->size = 1;
				break;

			case kPGPPrefType_Byte:
				if (IsntNull(member->data))
					PGPFreeData(member->data);

				member->data = NULL;
				member->size = 0;
				break;
			}
		}
		
		if (IsPGPError(err))
			break;
	}

	return err;
}


static PGPError sCopyStructDataToPref(PGPPrefRef prefRef, PGPBoolean readData,
					PGPPrefStruct *fromStruct, PGPPrefStruct *toStruct)
{
	PGPUInt32 index;
	PGPPrefStructMember *fromMember;
	PGPPrefStructMember *toMember;
	PGPMemoryMgrRef memoryMgr;
	PGPBoolean dirtyFlag;
	PGPError err = kPGPError_NoErr;

	memoryMgr = prefRef->memoryMgr;
	dirtyFlag = prefRef->dirty;

	if (IsNull(toStruct->members))
	{
		prefRef->dirty = TRUE;

		toStruct->members = (PGPPrefStructMember *) 
								PGPNewData(memoryMgr,
									sizeof(PGPPrefStructMember) * 
									fromStruct->numMembers,
									kPGPMemoryMgrFlags_Clear);
	}

	if (IsNull(toStruct->members))
		err = kPGPError_OutOfMemory;
	else
	{
		if (toStruct->numMembers != fromStruct->numMembers)
			prefRef->dirty = TRUE;

		if (toStruct->numMembers == 0)
			toStruct->numMembers = fromStruct->numMembers;

		for (index=0; index<fromStruct->numMembers; index++)
		{
			fromMember = &(fromStruct->members[index]);
			toMember = &(toStruct->members[index]);
		
			if (toMember->type == kPGPPrefType_String)
			{
				if (IsntNull(toMember->data))
					toMember->size = strlen(toMember->data) + 1;
				else
					toMember->size = 0;
			}

			if (fromMember->type == kPGPPrefType_String)
			{
				if (IsntNull(fromMember->data))
					fromMember->size = strlen(fromMember->data) + 1;
				else
					fromMember->size = 0;
			}

			if (toMember->type != fromMember->type)
			{
				prefRef->dirty = TRUE;
				toMember->type = fromMember->type;
			}
			
			if ((toMember->type != kPGPPrefType_Boolean) &&
				(toMember->type != kPGPPrefType_Number) &&
				(toMember->size != fromMember->size))
			{
				prefRef->dirty = TRUE;
				toMember->size = fromMember->size;
			}
			
			switch (toMember->type)
			{
			case kPGPPrefType_Boolean:
			case kPGPPrefType_Number:
				if (toMember->data != fromMember->data)
				{
					prefRef->dirty = TRUE;
					toMember->data = fromMember->data;
				}
				break;
				
			case kPGPPrefType_String:
				if (IsntNull(fromMember->data))
					toMember->size = strlen((char *) fromMember->data) + 1;
				else
					toMember->size = 0;
				
			case kPGPPrefType_Byte:
				if (IsntNull(toMember->data))
				{
					if (IsntNull(fromMember->data) && !prefRef->dirty)
					{
						if (!pgpMemoryEqual(fromMember->data, toMember->data,
							fromMember->size))
							prefRef->dirty = TRUE;
					}
					else
						prefRef->dirty = TRUE;
				}
				else if (IsntNull(fromMember->data))
					prefRef->dirty = TRUE;

				if (prefRef->dirty)
				{
					if (IsntNull(toMember->data))
					{
						PGPFreeData(toMember->data);
						toMember->data = NULL;
					}
					
					if (IsntNull(fromMember->data))
					{
						toMember->data = PGPNewData(memoryMgr,
											toMember->size,
											kPGPMemoryMgrFlags_Clear);
						
						if (IsNull(toMember->data))
							err = kPGPError_OutOfMemory;
						else
						{
							pgpCopyMemory(fromMember->data, toMember->data,
								toMember->size);

							if (toMember->type == kPGPPrefType_String)
							{
								sStripQuotes(toMember->size, toMember->data);
								toMember->size = strlen(toMember->data) + 1;
							}
						}
					}
				}
				break;
			}
			
			if (IsPGPError(err))
				break;
		}
	}

	if (readData)
		prefRef->dirty = dirtyFlag;

	return err;
}


static void sStripQuotes(PGPUInt32 stringLen, char *string)
{
	char *copyFrom;
	char *copyTo;
	PGPUInt32 index;

	index = 0;
	copyFrom = string;
	copyTo = string;
	while (index < stringLen)
	{
		if (copyFrom != copyTo)
			*copyTo = *copyFrom;

		if (*copyTo != '\"')
			copyTo++;

		copyFrom++;
		index++;
	};

	return;
}


static PGPError sIORead(PGPIORef ioRef, PGPSize dataSize, 
					PGPBoolean crossPlatform, void *data)
{
	PGPByte		tempData16[sizeof(PGPUInt16)];
	PGPByte		tempData32[sizeof(PGPUInt32)];
	PGPUInt32	data32;
	PGPUInt16	data16;
	PGPSize		bytesRead;
	PGPError	err = kPGPError_NoErr;

	PGPValidateIO(ioRef);
	PGPValidateParam(dataSize > 0);
	PGPValidatePtr(data);

	if (crossPlatform)
	{
		if (dataSize == sizeof(PGPUInt16))
			err = PGPIORead(ioRef, dataSize, tempData16, &bytesRead);
		else
			err = PGPIORead(ioRef, dataSize, tempData32, &bytesRead);

		if (IsntPGPError(err))
		{
			if (dataSize == sizeof(PGPUInt16))
			{
				data16 = PGPEndianToUInt16(kPGPBigEndian, tempData16);
				pgpCopyMemory(&data16, data, sizeof(PGPUInt16));
			}
			else
			{
				data32 = PGPEndianToUInt32(kPGPBigEndian, tempData32);
				pgpCopyMemory(&data32, data, sizeof(PGPUInt32));
			}
		}
	}
	else
		err = PGPIORead(ioRef, dataSize, data, &bytesRead);

	return err;
}


static PGPError sIOWrite(PGPIORef ioRef, PGPSize dataSize, 
					PGPBoolean crossPlatform, const void *data)
{
	PGPByte		tempData16[sizeof(PGPUInt16)];
	PGPByte		tempData32[sizeof(PGPUInt32)];
	PGPUInt32	data32;
	PGPUInt16	data16;
	PGPError	err = kPGPError_NoErr;

	PGPValidateIO(ioRef);
	PGPValidateParam(dataSize > 0);
	PGPValidatePtr(data);

	if (crossPlatform)
	{
		if (dataSize == sizeof(PGPUInt16))
		{
			pgpCopyMemory(data, &data16, sizeof(PGPUInt16));
			PGPUInt16ToEndian(data16, kPGPBigEndian, tempData16);
			err = PGPIOWrite(ioRef, dataSize, tempData16);
		}
		else
		{
			pgpCopyMemory(data, &data32, sizeof(PGPUInt32));
			PGPUInt32ToEndian(data32, kPGPBigEndian, tempData32);
			err = PGPIOWrite(ioRef, dataSize, tempData32);
		}
	}
	else
		err = PGPIOWrite(ioRef, dataSize, data);

	return err;
}


/*__Editor_settings____

	Local Variables:
	tab-width: 4
	End:
	vi: ts=4 sw=4
	vim: si
_____________________*/



